function err = diff_norm(vDp, unchanged)
% DIFF_NORM
% Computes the norm of the difference between two vectors
% Given tha data         : t = data(1,:) , y = data(2,:).
% and the FEM solution   : t, Cfo
%
% [t1 err] = diff_norm(vDp, unchanged)
%
% vDp		:= [v D beta t1 t2] 		% optimized parameters
% unchanged	:= {data, errbnd, output}	% non-optimized parameters
% v		:= velocity
% D		:= dispersion coefficient
% beta		:= TPL beta
% t1		:= TPL t1
% t2		:= TPL t2
% data		:= interpolated BTC 
% errbnd	:= the desired precision for fitting Prony parameters with sop2
% output	:= Name of output data file
% err		:= norm of the difference between the vectors 
% ------------------------------------------------------------
v	= vDp(1);
D	= vDp(2);
beta	= vDp(3);
t1	= vDp(4);
t2	= vDp(5);

data	= unchanged{1};
errbnd	= unchanged{2};
output	= unchanged{3};

fid = fopen(output,'a');

% fit Prony parameters to TPL
param = [beta log10(t1) log10(t2)]; % TPL parameters
[xs,ws,npols] = sop2(param,errbnd);

% ------------------------------ from mem.m ------------------------------
tmean = (((t2./t1).^beta) .* (-t1 + (exp(t1./t2)) .* ((t1./t2).^beta) .* (t1 + beta.*t2)...
	.*igamma(1 - beta, t1./t2)))./(exp(t1./t2).*(beta*igamma(-beta, t1./t2)));
tchar = t1;
tau2i= t1 ./ t2;
n =1./ (  tau2i.^beta .* exp(tau2i) .* igamma(-beta, tau2i)  );
% ------------------------------ from mem.m ------------------------------

Ao= n*tchar/t1; % Ao = n * tchar / t1
A = real(ws);	% Ap
B =-real(xs);	% Bp

fprintf(fid,'--- output from diff_norm --- \n');
fprintf(fid,'v \t = %14.7e \n', v);
fprintf(fid,'D \t = %14.7e \n', D);
fprintf(fid,'beta \t = %14.7e \n', beta);
fprintf(fid,'t1 \t = %14.7e \n', t1);
fprintf(fid,'t2 \t = %14.7e \n', t2);
fprintf(fid,'tmean \t = %14.7e \n', tmean);
fprintf(fid,'tchar \t = %14.7e \n', tchar);
fprintf(fid,'n \t = %14.7e \n', n);
fprintf(fid,'npols \t = %7i \n', npols);
fclose(fid);

%%%%%%%%%%% modify finp.txt %%%%%%%%%%%%
replaceLine = 1;
numLines = 100;	% >= actual No. of lines in the original finp.txt file

fid = fopen('finp.txt','r');
mydata = cell(1, numLines);
for k = 1:numLines
  mydata{k} = fgetl(fid);
  stat = feof(fid);
  if(stat == 1) break; end;
end
fclose(fid);

line = mydata{replaceLine};	% read line (Ne, Nn, Nb, Nm, Np)
val = sscanf(line, '%i');
Ne = val(1);
Nn = val(2);
Nb = val(3);
Nm = val(4);
Np = val(5);
s  = '! Ne, Nn, Nb, Nm, Np'; % the rest (string)
newtext = sprintf('%-i \t %-i \t %-i \t %-i \t %-i \t  %-s \t' ,Ne, Nn, Nb, Nm, npols, s); % replaced text
mydata{replaceLine} = newtext;

fid = fopen('finp.txt','w');
fprintf(fid, '%s \n', mydata{:});
fclose(fid);
%%%%%%%%%%% modify finp.txt %%%%%%%%%%%%

%%%%%%%%%%% write materials.txt %%%%%%%%%%%
fid = fopen('materials.txt','w');
fprintf(fid,'%14.7e		! Ao = beta \n', Ao);
for p=1:npols
  fprintf(fid,'%14.7e %14.7e	! par(p, 1:2)=Ap, Bp \n', A(p), B(p));
end	% p
fclose(fid);
%%%%%%%%%%% write materials.txt %%%%%%%%%%%

%%%%%%%%%%% write V.txt %%%%%%%%%%%
fid = fopen('V.txt','w');
for i=1:Nn
  fprintf(fid,'%10i %14.7e \n', i, v);
end	% p
fclose(fid);
%%%%%%%%%%% write V.txt %%%%%%%%%%%

%%%%%%%%%%% write D.txt %%%%%%%%%%%
fid = fopen('D.txt','w');
for i=1:Nn
  fprintf(fid,'%10i %14.7e \n', i, D);
end	% p
fclose(fid);
%%%%%%%%%%% write D.txt %%%%%%%%%%%

%%%%%%%%%%% run FEM %%%%%%%%%%%
[status cmdout] = dos('FEM-1D.exe');
% !!! continue only if status = 0 !!!

%%%%%%%%%%% run post_f_1 to get the BTC as t and Cfo arrays %%%%%%%%%%%
[t Cfo] = post_f_1();

%%%%%%%%%%% normalize Cfo
Cfo = Cfo * v*(n-1);

%%%%%%%%%%% calculate error (diffence between the BTC and the FEM solution)
err = 0;
Nt = length(t);

for k=1:Nt

% verify the time is identical (up to a tolerance)
  if ( abs( data(1,k) - t(k) ) > 1.e-10 )
     err = NaN;	% t is not identical
     return;
  end

% calculate only for numbers
  if (~isnan (data(2,k) ) )
    err = err + norm (Cfo(k) - data(2,k) );
  end
end
   
fid = fopen(output,'a');
fprintf(1,'difference = %6.4e \n',err);
fprintf(fid,'err \t = %14.7e \n', err);
fclose('all');  % close all files

